// IR-controlled RGB using IRremote v4.x (IrReceiver)
// Mapping required:
// 1 -> RED ON
// 2 -> RED OFF
// 3 -> GREEN ON
// 4 -> GREEN OFF
// 5 -> BLUE ON
// 6 -> BLUE OFF
// CH+ -> Increase brightness for ALL channels
// CH- -> Decrease brightness for ALL channels

#include <IRremote.h>

const int RECV_PIN = 2;

// RGB pins (use PWM pins)
const int RED_PIN   = 9;
const int GREEN_PIN = 10;
const int BLUE_PIN  = 11;

bool COMMON_ANODE = false; // set true if your RGB is common anode

// brightness values (0..255)
int redVal = 0;
int greenVal = 0;
int blueVal = 0;
const int STEP = 25; // brightness step

// ----- REPLACE THESE WITH YOUR REMOTE HEX CODES -----
// After running the scanner, paste the HEX values below (include 0x prefix).
// Example: const unsigned long CODE_1 = 0xFF6897;
const unsigned long CODE_1   = 0xF30CFF00; // BUTTON 1 - RED ON
const unsigned long CODE_2   = 0xE718FF00; // BUTTON 2 - RED OFF
const unsigned long CODE_3   = 0xA15EFF00; // BUTTON 3 - GREEN ON
const unsigned long CODE_4   = 0xF708FF00; // BUTTON 4 - GREEN OFF
const unsigned long CODE_5   = 0xE31CFF00; // BUTTON 5 - BLUE ON
const unsigned long CODE_6   = 0xA55AFF00; // BUTTON 6 - BLUE OFF
const unsigned long CODE_CHP = 0xEA15FF00; // VOL+ (increase)
const unsigned long CODE_CHM = 0xF807FF00; // VOL- (decrease)
// ----------------------------------------------------

void setup() {
  Serial.begin(115200);
  while (!Serial) {}
  Serial.println("IR RGB controller starting...");
  IrReceiver.begin(RECV_PIN, ENABLE_LED_FEEDBACK); // initialize receiver

  pinMode(RED_PIN, OUTPUT);
  pinMode(GREEN_PIN, OUTPUT);
  pinMode(BLUE_PIN, OUTPUT);

  applyRGB(); // make sure outputs initialized
}

int clampInt(int v, int lo, int hi) {
  if (v < lo) return lo;
  if (v > hi) return hi;
  return v;
}

void applyRGB() {
  // If common anode, invert values
  if (COMMON_ANODE) {
    analogWrite(RED_PIN,   255 - clampInt(redVal,   0, 255));
    analogWrite(GREEN_PIN, 255 - clampInt(greenVal, 0, 255));
    analogWrite(BLUE_PIN,  255 - clampInt(blueVal,  0, 255));
  } else {
    analogWrite(RED_PIN,   clampInt(redVal,   0, 255));
    analogWrite(GREEN_PIN, clampInt(greenVal, 0, 255));
    analogWrite(BLUE_PIN,  clampInt(blueVal,  0, 255));
  }

  Serial.print("RGB = ");
  Serial.print(redVal); Serial.print(", ");
  Serial.print(greenVal); Serial.print(", ");
  Serial.println(blueVal);
}

void loop() {
  if (IrReceiver.decode()) {
    // prefer decodedRawData for full raw value (works for many remotes)
    unsigned long code = IrReceiver.decodedIRData.decodedRawData;

    // Some remotes report repeat as 0xFFFFFFFF; ignore repeat-only values
    if (code == 0xFFFFFFFFUL) {
      IrReceiver.resume();
      return;
    }

    Serial.print("IR code: 0x");
    Serial.println(code, HEX);

    // ----- Mapping actions -----

    // RED ON / OFF
    if (code == CODE_1) {
      redVal = 255;
      Serial.println("RED ON");
    } else if (code == CODE_2) {
      redVal = 0;
      Serial.println("RED OFF");
    }

    // GREEN ON / OFF
    else if (code == CODE_3) {
      greenVal = 255;
      Serial.println("GREEN ON");
    } else if (code == CODE_4) {
      greenVal = 0;
      Serial.println("GREEN OFF");
    }

    // BLUE ON / OFF
    else if (code == CODE_5) {
      blueVal = 255;
      Serial.println("BLUE ON");
    } else if (code == CODE_6) {
      blueVal = 0;
      Serial.println("BLUE OFF");
    }

    // CH+ increase brightness for ALL channels
    else if (code == CODE_CHP) {
      redVal   = clampInt(redVal   + STEP, 0, 255);
      greenVal = clampInt(greenVal + STEP, 0, 255);
      blueVal  = clampInt(blueVal  + STEP, 0, 255);
      Serial.println("BRIGHTNESS +");
    }

    // CH- decrease brightness for ALL channels
    else if (code == CODE_CHM) {
      redVal   = clampInt(redVal   - STEP, 0, 255);
      greenVal = clampInt(greenVal - STEP, 0, 255);
      blueVal  = clampInt(blueVal  - STEP, 0, 255);
      Serial.println("BRIGHTNESS -");
    }

    applyRGB();

    IrReceiver.resume(); // ready for the next code
    delay(120);          // small debounce
  }
}
